//******************************************************************************
//  MSP430F20xx Demo - I2C Master Transmitter and Receiver for bq27210 EVM
//
//  Description: MSP430 I2C Master communicates with a bq27210EVM (I2C Slave)
//  using an I2C Master software driver.  Whenever master device transmits data
//  to the slave device, the data is read back for verification purposes.
//  If the data is incorrect, the program will toggle the P1.0 LED forever.
//  The various LED's on the bq27210EVM are enabled/disabled to demonstrate
//  I2C communications.
//
//  ACLK = n/a, MCLK = SMCLK = Calibrated 1MHz
//
//                   Slave                      Master
//                  bq27210                  MSP430F20xx
//             -----------------   /|\    -----------------
//            |                 |   |    |              XIN|-
//            |                 |   |    |                 |
//            |                 |   |    |             XOUT|-
//            |                 |  10K   |                 |
//            |                 |   |    |                 |
//            |                 |   |    |             P1.0|---> LED
//            |              SDA|---|--->|SDA              |
//            |              SCL|<--|----|SCL              |
//
//  Note: Internal pull-ups may be used in this example for SDA & SCL
//        bq27210 7-bit Slave address = 1010101b
//
//  R. Wu
//  Texas Instruments Inc.
//  January 2010
//  Built with IAR Embedded Workbench Version: 4.21A
//******************************************************************************

#include "bq27x10.h"                                 // Device-specific header
#include "..\..\lib\I2C_USI\USI_I2CMaster.h"         // For using the USI module
#include "..\..\lib\I2C_GPIO\MSP430_SWI2C_Master.h"  // For GPIO bit-banging

/**** USER CONFIGURATION: Select ONE of the following I2C physical layers *****/
#define _I2C_USI                            // OPTION #1: Use MSP430 USI module
//#define _I2C_BITBANG                        // OPTION #2: Use Standard GPIO pins

#define ATRATE_STEPS          100           // USER CONFIG: # of 3.57 uV steps
#define I2CSLAVEADDR         0x55           // 7-bit slave address
#define BUFFERSIZE              2           // # of bytes for Tx & Rx buffers

unsigned char TxData[BUFFERSIZE];           // Stores data bytes to be TX'd
unsigned char RxData[BUFFERSIZE];           // Stores data bytes that are RX'd
unsigned char rsoc;                         // Stores Relative State of Charge
unsigned char flags;                        // Stores Flags
unsigned char mode;                         // Stores Mode
unsigned int  temperature;                  // Stores temperature
unsigned int  voltage;                      // Stores voltage
unsigned int  atrate;                       // Stores AtRate
unsigned int  artte;                        // Stores AtRate Time to Empty
  signed int  offset;                       // Stores Offset measurement

int StatusCallback(unsigned char c);
unsigned int transBytes2Int(unsigned char msb, unsigned char lsb);
void MSP430_bq27210_read(unsigned char cmd, unsigned int bytes);
void MSP430_bq27210_write(unsigned char cmd, unsigned char data);
void MSP430_bq27210_error(void);

unsigned int transBytes2Int(unsigned char msb, unsigned char lsb)
{
  unsigned int tmp;
  
  tmp = ((msb << 8) & 0xFF00);
  return ((unsigned int)(tmp + lsb) & 0x0000FFFF);  
}

#ifdef _I2C_USI
int StatusCallback(unsigned char c)
{
 return TI_USI_EXIT_LPM;                    // Exit active for next transfer
}
#endif /* _I2C_USI */

void MSP430_bq27210_read(unsigned char cmd, unsigned int bytes)
{
  unsigned char tx[1];
  
  tx[0] = cmd;
#ifdef _I2C_USI
  __disable_interrupt();
  TI_USI_I2CWrite(I2CSLAVEADDR, 1, 1, tx);
  __bis_SR_register(LPM0_bits + GIE);
  __disable_interrupt();
  TI_USI_I2CRead(I2CSLAVEADDR, bytes, 0, RxData); 
  __bis_SR_register(LPM0_bits + GIE);
#endif /* _I2C_USI */
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_writeBlock(I2CSLAVEADDR, 1, 1, tx);
  MSP430_SWI2CMST_readBlock(I2CSLAVEADDR, bytes, RxData);
#endif /* _I2C_BITBANG */
}

void MSP430_bq27210_write(unsigned char cmd, unsigned char data)
{
  unsigned char tx[2];
  
  tx[0] = cmd;
  tx[1] = data;
#ifdef _I2C_USI
  __disable_interrupt();
  TI_USI_I2CWrite(I2CSLAVEADDR, 2, 0, tx);
  __bis_SR_register(LPM0_bits + GIE);
#endif /* _I2C_USI */
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_writeBlock(I2CSLAVEADDR, 2, 0, tx);
#endif /* _I2C_BITBANG */
  __delay_cycles(50000);
}

void MSP430_bq27210_error(void)
{
  while (1)                                 // Loop forever
  {
    P1OUT ^= BIT0;                          // Toggle LED
    __delay_cycles(50000);                  // Delay
  }
}

void main(void)
{
  unsigned char msb, lsb;
  
  WDTCTL = (WDTPW | WDTHOLD);               // Disable Watchdog
  BCSCTL1 = CALBC1_1MHZ;                    // Set DCO to 1 MHz
  DCOCTL = CALDCO_1MHZ;
  P1DIR |= BIT0;                            // P1.0 output direction
  P1OUT &= ~BIT0;                           // Turn P1.0 LED OFF
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_init();                   // Initialize I2C pins
#endif /* _I2C_BITBANG */
  
#ifdef _I2C_USI
  // Initialize USI module, clock = ~(SMCLK/32) 
  TI_USI_I2C_MasterInit((USIDIV_5 | USISSEL_2 | USICKPL), StatusCallback);
  
  // Acknowledge polling function - LED blinks continuously until slave device 
  // provides an ACK
  while (TI_USI_I2CSelect(I2CSLAVEADDR))
  { 
    P1OUT ^= BIT0;                          // Toggle LED
    __delay_cycles(50000);                  // Delay    
  }
  P1OUT = 0x00;                             // Slave acknowledged, LED off
#endif /* _I2C_USI */
 
  // Read Status Flags
  MSP430_bq27210_read(bq27x10CMD_FLAGS, 1);
  flags = RxData[0];
 
  // GPIO open drain output - Turn ON
  MSP430_bq27210_write(bq27x10CMD_CTRL, 0xA9); // Can use any of the 3 cmd keys
  MSP430_bq27210_read(bq27x10CMD_MODE, 1);
  mode = RxData[0];
  mode &= ~(bq27x10REG_MODE_GPIEN|bq27x10REG_MODE_GPSTAT);
  MSP430_bq27210_write(bq27x10CMD_MODE, mode);
  MSP430_bq27210_read(bq27x10CMD_MODE, 1);
  if (mode != RxData[0])
  {
    MSP430_bq27210_error();                 // Signal error condition occurred    
  }

  // Read Relative State of Charge (units = %)
  MSP430_bq27210_read(bq27x10CMD_RSOC, 1);
  rsoc = RxData[0];
  
  // Read Voltage (units = mV)
  MSP430_bq27210_read(bq27x10CMD_VOLT_LSB, 2);  
  voltage = transBytes2Int(RxData[1], RxData[0]);

  // Read Temperature (units = 0.25K)
  MSP430_bq27210_read(bq27x10CMD_TEMP_LSB, 2);  
  temperature = transBytes2Int(RxData[1], RxData[0]);

  // Set # of steps for AtRate (1 step = 3.57 uV)
  atrate = ATRATE_STEPS;
  msb = ((atrate >> 8) & 0x00FF);
  lsb = (atrate & 0x00FF);
  MSP430_bq27210_write(bq27x10CMD_AR_MSB, msb);
  MSP430_bq27210_write(bq27x10CMD_AR_LSB, lsb);
  
  // Verify AtRate was set correctly
  MSP430_bq27210_read(bq27x10CMD_AR_LSB, 2); 
  if ((RxData[1] != msb) || (RxData[0] != lsb))
  {
    MSP430_bq27210_error();                 // Signal error condition occurred
  }
  
  // Read AtRate Time to Empty (units = Minutes)
  MSP430_bq27210_read(bq27x10CMD_ARTTE_LSB, 2); 
  artte = transBytes2Int(RxData[1], RxData[0]);
   
  // Perform internal offset measurement
  MSP430_bq27210_write(bq27x10CMD_MODE, bq27x10REG_MODE_CIO);
  MSP430_bq27210_write(bq27x10CMD_CTRL, 0x56);
  
  // Monitor Calibration-In-Progress flag until offset measurement is done
  RxData[0] = bq27x10REG_FLAGS_CALIP;
  while (RxData[0] & bq27x10REG_FLAGS_CALIP)
  {
    MSP430_bq27210_read(bq27x10CMD_FLAGS, 1);
  }

  // Read offset measurement value
  MSP430_bq27210_read(0x5E, 2); 
  offset = transBytes2Int(RxData[1], RxData[0]);
 
  // GPIO open drain output - Turn OFF
  MSP430_bq27210_write(bq27x10CMD_CTRL, 0xC5); // Can use any of the 3 cmd keys
  MSP430_bq27210_read(bq27x10CMD_MODE, 1);
  mode = RxData[0];
  mode |= (bq27x10REG_MODE_GPIEN|bq27x10REG_MODE_GPSTAT);
  MSP430_bq27210_write(bq27x10CMD_MODE, mode);
  MSP430_bq27210_read(bq27x10CMD_MODE, 1);
  if (mode != RxData[0])
  {
    MSP430_bq27210_error();                 // Signal error condition occurred    
  }

  // Set P1.0 LED on MSP430 EVM to signal that command sequence was successful
  P1OUT |= BIT0; 
}
